/************************************************************************************************************\

Module Name:    EncoderModule.h

Description:    Module for Liberatus H264 video encoder.

    Copyright (c) 2015, Matrox Graphics Inc. All Rights Reserved.

    BSD 2-Clause License

    Redistribution and use in source and binary forms, with or without modification, are permitted provided
    that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this list of conditions and the
       following disclaimer.

    2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
       the following disclaimer in the documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
    WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR
    ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
    INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
    TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
    ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

\************************************************************************************************************/

#ifndef INC_EncoderModule_H
#define INC_EncoderModule_H

// -----------------------------------------------------------------------------------------------------------
//                                   I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------

#include "Liberatus.h"
#include "CommonUtils.h"
#include "ModuleThread.h"
#include "ModuleLink.h"
#include "LH264VideoCodec.h"
#include "LVideoProcessor.h"
#include "LBlit.h"
//#include "MtxBaseMacros.h"


// -----------------------------------------------------------------------------------------------------------
//                                   C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------

#define ENC_MAX_BITRATE_KBPS    (900*1000)
#define ENC_MAX_NB_SLICES       12
#define ENC_WIDTH_ALIGN         16
#define ENC_HEIGHT_ALIGN        16

// Performance problems when we schedule a wait submission into the device thread of the encoder input.
#define ENC_ENABLE_SYNC_CPU_PATCH   MTRUE

/************************************************************************************************************\

Structure:      EncoderModule

Description:    Module for Liberatus H264 video encoder.

\************************************************************************************************************/
typedef struct tagEncoderModule
{
    LDeviceThread_Handle    hDevThreadIn;       // Input device thread
    LDeviceThread_Handle    hDevThreadOut;      // Output device thread
    LDeviceThread_Handle    hDevThreadPad;      // Device thread used to draw padding.
    LDeviceThread_Handle    hDevThreadCopy;     // Device thread used to copy NALUs to output buffers.
    LH264E_Handle           hEncoder;           // Encoder handle
    LVPE_Handle             hVpe;               // VPE handle used to draw padding.
    LBlit_Handle            hBlit;              // Blit handle to copy NALUS to output buffers.
    ModuleThread            oCpuThreadIn;       // Input CPU thread
    ModuleThread            oCpuThreadOut;      // Output CPU thread
    ModuleLinkInput         oInLink;            // Input module link
    ModuleLink              oOutLink;           // Output module link
    LSIZE                   oFrameSize;         // Frame size.
    LSIZE                   oFramePadSize;      // Padding size of the input frame.
    MBOOL                   bEncoding;          // Indicate if the encoder is running.
    MUINT64                 uiElapsedTimeUsec;  // Elapse time elpase since the first completed NALU in usec.
    MUINT                   uiNaluCount;        // Number of completed NALUs.
    MUINT                   uiPictureCount;     // Number of completed pictures.
    MUINT64                 uiEncodedBytes;     // Number of bytes encoded.
    MVOID*                  pvPrivData;         // Private data.
    MCHAR8                  szModuleName[16];

    // Encoding options

    MUINT                                   uiNbSlices;
    LH264_Profile                           eProfile;
    LH264_Level                             eLevel;
    LH264E_EncodeOpt                        oEncodeOpt;
    LH264_StreamInfoEncoderSequenceData     oSeqData;
    LH264_StreamInfoEncoderSubSequenceData  oSubSeqData;

} EncoderModule;

#define EncoderModule_Construct { /*.hDevThreadIn       =*/ MNULL, \
                                  /*.hDevThreadOut      =*/ MNULL, \
                                  /*.hDevThreadVpe      =*/ MNULL, \
                                  /*.hDevThreadCopy     =*/ MNULL, \
                                  /*.hEncoder           =*/ MNULL, \
                                  /*.hVpe               =*/ MNULL, \
                                  /*.hBlit              =*/ MNULL, \
                                  /*.oCpuThreadIn       =*/ ModuleThread_Construct, \
                                  /*.oCpuThreadOut      =*/ ModuleThread_Construct, \
                                  /*.oInLink            =*/ ModuleLinkInput_Construct, \
                                  /*.oOutLink           =*/ ModuleLink_Construct, \
                                  /*.oFrameSize         =*/ {0, 0}, \
                                  /*.oFramePadSize      =*/ {0, 0}, \
                                  /*.bEncoding          =*/ MFALSE, \
                                  /*.uiElapsedTimeUsec  =*/ 0, \
                                  /*.uiNaluCount        =*/ 0, \
                                  /*.uiPictureCount     =*/ 0, \
                                  /*.uiEncodedBytes     =*/ 0, \
                                  /*.pvPrivData         =*/ MNULL, \
                                  /*.szModuleName       =*/ "",\
                                  /*.uiNbSlices         =*/ 1 }

// -----------------------------------------------------------------------------------------------------------
//                  G L O B A L   V A R I A B L E / F U N C T I O N   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------
LStatus EncMod_Init(EncoderModule*      poEncMod,
                    LDevice_Handle      hDevice,
                    MUINT               uiOutBufferCount,
                    LBuffer_Attributes* poOutBufferAttributes,
                    LH264E_EncoderMode  eEncodeMode,
                    LH264_Profile       eProfile,
                    LH264_Level         eLevel,
                    MUINT               uiSliceCount,
                    MUINT               uiFaq,
                    MBOOL               bDfEnabled);
void EncMod_Cleanup(EncoderModule* poEncMod);
LStatus EncMod_SetParam(
        EncoderModule* poEncMod,
        LSIZE*  poFrameSize,
        MUINT   uiFrameRateNum,
        MUINT   uiFrameRateDen,
        MUINT   uiTargetBitRateKbps,
        MUINT   uiIPeriod,
        MUINT   uiPPeriod,
        MUINT   uiQpMin,
        MUINT   uiQpMax);
LStatus EncMod_Start(EncoderModule* poEncMod);
void EncMod_Stop(EncoderModule* poEncMod);

// -----------------------------------------------------------------------------------------------------------
//                                 I N L I N E S   D E F I N I T I O N S
// -----------------------------------------------------------------------------------------------------------
static __inline void EncMod_GetPaddedFrameSize(const LSIZE* poSizeIn, LSIZE* poSizeOut)
{
    poSizeOut->iWidth  = AlignPow2(poSizeIn->iWidth,  ENC_WIDTH_ALIGN);
    poSizeOut->iHeight = AlignPow2(poSizeIn->iHeight, ENC_HEIGHT_ALIGN);
}

#endif // INC_EncoderModule_H
